<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
require_once __DIR__ . '/../vendor/autoload.php';

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Cargar variables de entorno desde .env
$dotenv = Dotenv::createImmutable(__DIR__ . '/../');
$dotenv->load();

$host = $_ENV['DB_HOST'] ?? 'localhost';
$dbname = $_ENV['DB_NAME'] ?? '';
$username = $_ENV['DB_USER'] ?? '';
$password = $_ENV['DB_PASS'] ?? '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode(["error" => "Error de conexión a la base de datos"]);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['first_name']) || !isset($input['last_name']) || 
        !isset($input['email']) || !isset($input['birth_date'])) {
        http_response_code(400);
        echo json_encode(["error" => "Faltan datos requeridos"]);
        exit();
    }
    
    $firstName = trim($input['first_name']);
    $lastName = trim($input['last_name']);
    $email = trim($input['email']);
    $birthDate = $input['birth_date'];
    
    // Validaciones básicas
    if (empty($firstName) || empty($lastName) || empty($email)) {
        http_response_code(400);
        echo json_encode(["error" => "Todos los campos son requeridos"]);
        exit();
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(["error" => "Email no válido"]);
        exit();
    }
    
    // Verificar si el email ya existe
    $checkEmail = $pdo->prepare("SELECT id FROM bo_users WHERE email = ?");
    $checkEmail->execute([$email]);
    
    if ($checkEmail->fetch()) {
        http_response_code(409);
        echo json_encode(["error" => "Este email ya está registrado"]);
        exit();
    }
    
    try {
        // Llamar al procedimiento almacenado RegisterUser
        $stmt = $pdo->prepare("CALL RegisterUser(?, ?, ?, ?, @user_id, @temp_password)");
        $stmt->execute([$firstName, $lastName, $email, $birthDate]);
        
        // Obtener los valores de salida
        $result = $pdo->query("SELECT @user_id as user_id, @temp_password as temp_password")->fetch(PDO::FETCH_ASSOC);
        
        if ($result && $result['user_id']) {
            // Obtener datos completos del usuario
            $userQuery = $pdo->prepare("
                SELECT id, first_name, last_name, email, birth_date, 
                       created_at, is_email_verified, interests
                FROM bo_users 
                WHERE id = ?
            ");
            $userQuery->execute([$result['user_id']]);
            $userData = $userQuery->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                "success" => true,
                "message" => "Usuario registrado exitosamente",
                "temporary_password" => $result['temp_password'],
                "user" => [
                    "id" => $userData['id'],
                    "first_name" => $userData['first_name'],
                    "last_name" => $userData['last_name'],
                    "email" => $userData['email'],
                    "birth_date" => $userData['birth_date'],
                    "created_at" => $userData['created_at'],
                    "is_email_verified" => (bool)$userData['is_email_verified'],
                    "interests" => $userData['interests'] ?? ""
                ]
            ]);
        } else {
            http_response_code(500);
            echo json_encode(["error" => "Error al registrar usuario"]);
        }
        
    } catch(PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Error del servidor: " . $e->getMessage()]);
    }
    
} else {
    http_response_code(405);
    echo json_encode(["error" => "Método no permitido"]);
}
?>
